﻿/*
VERSION: 1.1
1.1		Added "change" event

USAGE:
	#include "functions/tank_turn.as"
	player_mc.turn_obj = makeTankTurn( player_mc );											// this handles rotation of the "radian"
	#include "functions/tank_move.as"
	player_mc.move_obj = makeTankMove( player_mc, player_mc.sprite );		// this moves the player
	
	player_mc.turn_obj.resetVelocity();		// set velocity to Zero		(useful when switching between movement systems in real-time)
	player_mc.move_obj.resetVelocity();		// set velocity to Zero		(useful when switching between movement systems in real-time)
	
	// tank_turn will emit "change_radian" events within player_mc
	// tank_move will detect those events within player_mc and update its own radian
	
	onUnload = function(){
		clearInterval(turn_obj.loopInterval);
		clearInterval(move_obj.loopInterval);
	}

	
SETTINGS:
	key_moveForward		Key code for moving forward
	key_moveBackward	Key code for moving backward
	
	xVel					previously applied horizontal speed		(in pixels per frame)
	yVel					previously applied vertical speed			(in pixels per frame)
	
PROPERTIES:
	maxSpeed			Maximum speed  (in pixels per frame)
	acel					Acelleration speed,  how many pixels the velocity increases per frame while a key is pressed
	decel					Deceleration speed, multiplier that reduces velocity per frame when no key is pressed
	walkSpeed			Maximum speed  (in pixels per frame)
	
	
RAM.keys defaults
	RAM.keys.up
	RAM.keys.down
	

EVENTS: 
	unload
	change				{value: #}		(+ forward  - backward)
	change_radian		{value: Point}		(local-relative movement.  -y forward  +y backward  -x left  +x right)			Only fires within player
	
PLAYER EVENTS: 
	request_radian
	
RESPONDS TO PLAYER EVENTS:
	unload
	change_radian
	
	
NOTES:
	This system has optional hooks for these collision and sprite systems:
		sprite.as
		WalkCollision.as
	It will use them if they're present.
	
	This system relies on a "radian" variable,  but cannot alter it by itself.
		This variable can be manually altered by changing the value of  move_obj.radian
		or it can be automatically updated via  "change_radian"  events firing within player_mc or move_obj
	
	
ADDING COLLISION:
	A collision system can be added like this:
		move_obj.collision = new Collision( player_mc, move_obj, collision_array, -8, -8 );
	The move_obj is this movement system.
*/
function makeTankMove( player_mc, playerSprite_mc ){
	#include "eventSystem3.as"
	var _this = {};		// external interface
	AsBroadcaster.initialize( _this );
	var react = make_react( _this );		// param is optional
	react.to("unload").from( player_mc ).tell( _this );
	react.to("change_radian").from( player_mc ).tell( _this );
	
	
	
	// Variables
	var fps = 30;
	// movement speed
	_this.move_acel = 1;				// acelleration increment
	_this.move_decel = 0.6;			// brake multiplier
	_this.move_speed = 0;				// current movement speed
	_this.move_maxSpeed = 10;
	// cartesian velocities for:   external collision systems  =>  final movement
	_this.xVel = 0;
	_this.yVel = 0;
	// controls
	_this.key_moveForward = RAM.keys.up || Key.UP;
	_this.key_moveBackward = RAM.keys.down || Key.DOWN;
	// allow the player's image-sprite to be swapped-out later
	_this.playerSprite_mc = playerSprite_mc;
	// rotation
	_this.radian = Math.PI /180 *90;		// current rotation  (in radians)
	// initially request a radian from player_mc,  and store the result
	sendEvent( "request_radian", {callback: function( newValue ){
		_this.radian = newValue;
	}}, player_mc );
	// update whenever external code changes the radian
	react.to("change_radian").then = function( evt ){
		_this.radian = evt.value;
	}
	/*	
	react.to("change_radian").from( player_mc ).then = function( evt ){
		_this.radian = evt.value;
	}
	*/
	
	
	// legacy
	// // move speed
	_this.addProperty( "maxSpeed", function(){
		return _this.move_maxSpeed;
	}, function( newValue ){
		_this.move_maxSpeed = newValue;
	});
	_this.addProperty( "walkSpeed", function(){
		return _this.move_maxSpeed;
	}, function( newValue ){
		_this.move_maxSpeed = newValue;
	});
	_this.addProperty( "acel", function(){
		return _this.move_acel
	}, function( newValue ){
		_this.move_acel = newValue;
	});
	_this.addProperty( "decel", function(){
		return _this.move_decel
	}, function( newValue ){
		_this.move_decel = newValue;
	});
	// // controls
	_this.addProperty( "up_key", function(){
		return _this.key_moveForward
	}, function( newValue ){
		_this.key_moveForward = newValue;
	});
	_this.addProperty( "down_key", function(){
		return _this.key_moveBackward
	}, function( newValue ){
		_this.key_moveBackward = newValue;
	});
	
	
	
	_this.resetVelocity = function(){
		_this.move_speed = 0;
	}// resetVelocity()
	
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	// Main Program
	var lastMoveSpeed = 0;
	_this.loop = function(){
		//  Abort if the player sprite doesn't exist
		if(player_mc._currentframe === undefined){
			sendEvent("unload");
			return;
		}// if:   player_mc is missing
		
		// update total speeds
		_this.move_speed = changeMoveVelocity( _this.move_speed );		// pixel movement magnitude
		_this.move_speed = limitMoveSpeed( _this.move_speed );
		
		// get cartesian velocities
		var velocity_vec = get_velocity_vec( _this.move_speed, _this.radian );
		_this.xVel = velocity_vec.x;
		_this.yVel = velocity_vec.y;
		
		_this.collision.run();		// apply external collision  (zeros xVel/yVel when collisions occur)
		setAnimSpeed( _this.xVel, _this.yVel );
		applyMovement( _this.xVel, _this.yVel );
		
		if( lastMoveSpeed !== _this.move_speed )		sendEvent( "change", {value: _this.move_speed});
		lastMoveSpeed = _this.move_speed;
	}// loop()
	_this.loopInterval = setInterval( function(){
		_this.loop();
	}, 1000/fps );
	react.to("unload").then = function(){
		clearInterval(_this.loopInterval);
	}
	
	return _this;
	
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	function limitMoveSpeed( move_speed ){
		if( move_speed > _this.move_maxSpeed )		move_speed = _this.move_maxSpeed;
		if( move_speed < -_this.move_maxSpeed )		move_speed = -_this.move_maxSpeed;
		return move_speed;
	}// limitMoveSpeed()
	
	
	
	function decellerateMoveSpeed( move_speed ){
		move_speed *= _this.move_decel;
		if( Math.abs( move_speed ) < 1 )		move_speed = 0;		// if less than 1 pixel of movement,  then => no movement
		return move_speed;
	}// decellerateMoveSpeed()
	
	
	
	function applyMovement( x, y ){
		player_mc._x += x;
		player_mc._y += y;
	}// applyMovement()
	
	
	
	function setAnimSpeed( xVel, yVel ){
		var xVel = Math.abs(xVel);
		var yVel = Math.abs(yVel);
		// adjust walk speed
		var max = 5;
		var range = 4;
		var topSpeed = 10;		// used for animation
		//var moveSpeed = Math.sqrt(_this.xVel*_this.xVel + _this.yVel*_this.yVel);
		var moveSpeed =  (xVel > yVel)  ?  xVel  :  yVel;		// Use the fastest velocity
		var speedPercent = Math.abs(moveSpeed) / topSpeed;		// get percent
		var animSpeed = max - (range * speedPercent);
		var animSpeed = (animSpeed<1) ? 1 : animSpeed;		// enforce max speed
		_this.playerSprite_mc.setParams({delay:animSpeed});
	}// setAnimSpeed()
	
	
	
	function get_velocity_vec( moveSpeed, turnSpeed ){
		var Point = flash.geom.Point;
		var velocity_vec = Point.polar( -moveSpeed, turnSpeed );
		return velocity_vec;
	}// get_velocity_vec()
	
	
	
	function changeMoveVelocity( move_speed ){
		// move
		if( _this.key_moveForward  &&  Key.isDown( _this.key_moveForward ) ){
			if( move_speed < _this.move_maxSpeed ){
				if(move_speed < 0)		decellerateMove();
				move_speed += _this.move_acel;
			}// if:   not at or beyond the max speed
		}else if( _this.key_moveBackward  &&  Key.isDown( _this.key_moveBackward ) ){
			if( move_speed > -_this.move_maxSpeed ){
				if(turn_vel > 0)		decellerateMove();
				move_speed -= _this.move_acel;
			}// if:   not at or beyond the max speed
		}else{
			decellerateMove();
		}
		function decellerateMove(){
			move_speed *= _this.move_decel;
			if( Math.abs( move_speed ) < 1 )		move_speed = 0;		// if less than 1 pixel of movement,  then => no movement
		}// decellerateMove()
		
		if( move_speed > _this.move_maxSpeed )		move_speed = _this.move_maxSpeed;
		if( move_speed < -_this.move_maxSpeed )		move_speed = -_this.move_maxSpeed;
		
		return move_speed;
	}// changeMoveVelocity()
	
}// makeTankMove()